let pdfDoc = null;
let pagesData = []; // This holds the state of every page
let sortableInstance = null;
let selectedPageId = null;

const gridContent = document.getElementById("gridContent");
const fileInput = document.getElementById("fileInput");
const dropZone = document.getElementById("dropZone");
const rangeInput = document.getElementById("pageRangeInput");

// --- 1. Initialization ---
fileInput.onchange = (e) => handleFile(e.target.files[0]);

async function handleFile(file) {
  if (!file) return;
  document.getElementById("loadingOverlay").classList.remove("hidden");

  try {
    const arrayBuffer = await file.arrayBuffer();
    pdfDoc = await pdfjsLib.getDocument({ data: arrayBuffer }).promise;

    pagesData = [];
    gridContent.innerHTML = "";

    for (let i = 1; i <= pdfDoc.numPages; i++) {
      pagesData.push({ id: i, rotation: 0, isRemoved: false });
      await renderThumbnail(i);
    }

    // --- UPDATED: Sortable is now ALWAYS enabled ---
    sortableInstance = new Sortable(gridContent, {
      animation: 150,
      ghostClass: "sortable-ghost",
      disabled: false, // <--- CHANGED TO FALSE (Always On)
      delay: 0,
      onEnd: () => {
        syncDataOrder(); // Update internal array immediately after drop
      },
    });

    document.getElementById("step-upload").classList.add("hidden");
    document.getElementById("step-editor").classList.remove("hidden");
    document.getElementById("totalCount").innerText = pdfDoc.numPages;
  } catch (err) {
    console.error(err);
    alert("Error loading PDF");
  } finally {
    document.getElementById("loadingOverlay").classList.add("hidden");
  }
}

async function renderThumbnail(pageNum) {
  const page = await pdfDoc.getPage(pageNum);
  const viewport = page.getViewport({ scale: 0.3 });
  const canvas = document.createElement("canvas");
  const context = canvas.getContext("2d");
  canvas.height = viewport.height;
  canvas.width = viewport.width;

  await page.render({ canvasContext: context, viewport: viewport }).promise;

  const card = document.createElement("div");
  // Added 'cursor-move' class here so it always looks draggable
  card.className = "page-card"; 
  card.dataset.id = pageNum;
  card.innerHTML = `
        <div class="remove-badge" onclick="toggleRemove(${pageNum})">✕</div>
        <div class="rotate-badge" onclick="rotatePage(${pageNum})">
            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/></svg>
        </div>
        <div class="page-inner">
            <img src="${canvas.toDataURL()}" class="thumb-img">
            <div class="page-label">Page ${pageNum}</div>
        </div>
    `;
  gridContent.appendChild(card);
}

// --- 2. State Management ---

function updateUI() {
  let removedCount = 0;
  const rangeArray = parseRange(rangeInput.value);

  pagesData.forEach((p) => {
    const card = document.querySelector(`[data-id="${p.id}"]`);
    if (!card) return;

    // Check if removed by Click OR by Range Input
    if (p.isRemoved || rangeArray.includes(p.id)) {
      card.classList.add("to-remove"); // Ensure you have CSS for .to-remove (e.g., opacity: 0.4)
      removedCount++;
    } else {
      card.classList.remove("to-remove");
    }
  });
  document.getElementById("removeCount").innerText = removedCount;
}

function toggleRemove(id) {
  const page = pagesData.find((p) => p.id === id);
  if (page) {
    page.isRemoved = !page.isRemoved;
    updateUI();
  }
}

rangeInput.addEventListener("input", updateUI);

function parseRange(input) {
  const pages = [];
  if (!input.trim()) return [];
  input.split(",").forEach((part) => {
    if (part.includes("-")) {
      const [start, end] = part.split("-").map(Number);
      if (!isNaN(start) && !isNaN(end)) {
        for (let i = Math.min(start, end); i <= Math.max(start, end); i++)
          pages.push(i);
      }
    } else {
      const num = parseInt(part.trim());
      if (!isNaN(num)) pages.push(num);
    }
  });
  return pages;
}

function rotatePage(id) {
  const page = pagesData.find((p) => p.id === id);
  page.rotation = (page.rotation + 90) % 360;
  const img = document.querySelector(`[data-id="${id}"] .thumb-img`);
  img.style.transform = `rotate(${page.rotation}deg)`;
}

function resetAll() {
  pagesData.forEach((p) => {
    p.isRemoved = false;
    p.rotation = 0;
    const img = document.querySelector(`[data-id="${p.id}"] .thumb-img`);
    if(img) img.style.transform = `rotate(0deg)`;
  });
  rangeInput.value = "";
  // Optional: Reset order logic here if you want to undo reordering too
  updateUI();
}

// --- 3. Reordering Logic (Always Active) ---

// Syncs the pagesData array with what the user sees on screen
function syncDataOrder() {
  const newOrderIds = Array.from(gridContent.children).map((c) => parseInt(c.dataset.id));
  pagesData.sort((a, b) => newOrderIds.indexOf(a.id) - newOrderIds.indexOf(b.id));
}

// Click to Select
gridContent.addEventListener("click", (e) => {
  const card = e.target.closest(".page-card");
  if (!card) return;
  if (e.target.closest(".remove-badge") || e.target.closest(".rotate-badge")) return;

  document.querySelectorAll(".page-card").forEach((c) => c.classList.remove("ring-4", "ring-indigo-500"));
  card.classList.add("selected");
  selectedPageId = parseInt(card.dataset.id);
});

// Button Movement (Top, End, Up, Down)
function moveSelected(direction) {
  if (selectedPageId === null) {
    alert("Please click a page to select it first.");
    return;
  }
  const card = document.querySelector(`[data-id="${selectedPageId}"]`);
  if (!card) return;

  const parent = gridContent;
  if (direction === "start") parent.prepend(card);
  else if (direction === "end") parent.appendChild(card);
  else if (direction === "up" && card.previousElementSibling) parent.insertBefore(card, card.previousElementSibling);
  else if (direction === "down" && card.nextElementSibling) parent.insertBefore(card.nextElementSibling, card);

  syncDataOrder(); // Sync immediately after button move
}

function buildPageManifest() {
  return pagesData.map((p, index) => ({
    pageNumber: p.id,        // original page number
    order: index + 1,        // new order
    rotation: p.rotation,   // 0, 90, 180, 270
    removed: p.isRemoved    // true / false
  }));
}

// ===================== APPLY & PROCESS =====================


async function applyAndDownload() {
  if (!pdfDoc) {
    alert("No PDF loaded");
    return;
  }

  try {
    document.getElementById("loadingOverlay").classList.remove("hidden");
    syncRangeRemovalToState();
    const formData = new FormData();

    // 1. Original PDF file
    formData.append("file", fileInput.files[0]);

    // 2. Page Manifest
    formData.append(
      "pages",
      JSON.stringify(buildPageManifest())
    );

    // 3. Output filename
    const filename =
      document.getElementById("outputFilename").value.trim() || "PDF_NEX";
    formData.append("filename", filename);

    // 4. Send to backend
    const res = await fetch("/remove_pages", {
      method: "POST",
      body: formData,
    });

    if (!res.ok) throw new Error("Processing failed");

    // 5. Receive PDF
    const blob = await res.blob();
    const url = URL.createObjectURL(blob);

    // 6. Setup download button
    const downloadBtn = document.getElementById("downloadBtn");
    downloadBtn.href = url;
    downloadBtn.download = filename + ".pdf";

    // 7. Setup share buttons
    // setupShareButtons(url, filename);

    // 8. Switch screen
    document.getElementById("step-editor").classList.add("hidden");
    document.getElementById("step-success").classList.remove("hidden");

  } catch (err) {
    console.error(err);
    alert(err.message);
  } finally {
    document.getElementById("loadingOverlay").classList.add("hidden");
  }
}
document.getElementById("applyBtn").addEventListener("click", applyAndDownload);
function syncRangeRemovalToState() {
  const rangeArray = parseRange(rangeInput.value);

  pagesData.forEach(p => {
    if (rangeArray.includes(p.id)) {
      p.isRemoved = true;
    }
  });
}
