let draggedPageId = null;
let isDraggingPage = false;
let draggedFileId = null;
let isDraggingFile = false;

// --- STATE MANAGEMENT ---
const state = {
  files: [],
  pages: [],
  selectedIds: new Set(),
  viewMode: "pages", // 'files' or 'pages'
  zoomLevel: 1,
  quality: 4,
};

// --- DOM ELEMENTS ---
const els = {
  dropZone: document.getElementById("dropZone"),
  fileInput: document.getElementById("fileInput"),
  stepUpload: document.getElementById("step-upload"),
  stepEditor: document.getElementById("step-editor"),
  stepSuccess: document.getElementById("step-success"),
  gridContent: document.getElementById("gridContent"),
  loadingOverlay: document.getElementById("loadingOverlay"),
  itemCount: document.getElementById("itemCount"),
  applyBtn: document.getElementById("applyBtn"),
  downloadBtn: document.getElementById("downloadBtn"),
  outputFilename: document.getElementById("outputFilename"),
  btnViewFiles: document.getElementById("btnViewFiles"),
  btnViewPages: document.getElementById("btnViewPages"),
  zoomControls: document.getElementById("zoomControls"),
  qualityRange: document.getElementById("qualityRange"),
  qualityValue: document.getElementById("qualityValue"),
  qualityDesc: document.getElementById("qualityDesc"),
  editTools: document.getElementById("editTools"),
  shareEmail: document.getElementById("shareEmail"),
  shareWhatsapp: document.getElementById("shareWhatsapp"),
};

// --- INITIALIZATION ---
els.dropZone.addEventListener("dragover", (e) => {
  e.preventDefault();
  els.dropZone.classList.add("drag-over");
});
els.dropZone.addEventListener("dragleave", () =>
  els.dropZone.classList.remove("drag-over")
);
els.dropZone.addEventListener("drop", (e) => {
  e.preventDefault();
  els.dropZone.classList.remove("drag-over");
  handleFiles(e.dataTransfer.files);
});
els.fileInput.addEventListener("change", (e) => handleFiles(e.target.files));

// Quality Slider Logic
const qualityLabels = {
  1: { label: "Low", desc: "Smallest size, lower visual quality" },
  2: { label: "Medium", desc: "Good balance for web sharing" },
  3: { label: "High", desc: "High quality, optimized compression" },
  4: { label: "Original", desc: "No compression, original quality" },
};

els.qualityRange.addEventListener("input", (e) => {
  state.quality = parseInt(e.target.value);
  const q = qualityLabels[state.quality];
  els.qualityValue.innerText = q.label;
  els.qualityDesc.innerText = q.desc;
});

// --- FILE HANDLING ---
async function handleFiles(fileList) {
  const newFiles = Array.from(fileList).filter(
    (f) => f.type === "application/pdf"
  );
  if (newFiles.length === 0) return alert("PDF only");

  showLoading();

  for (let file of newFiles) {
    const arrayBuffer = await file.arrayBuffer();
    const pdf = await pdfjsLib.getDocument(arrayBuffer).promise;
    const fileId = `f-${Date.now()}-${Math.random()}`;

    state.files.push({
      id: fileId,
      fileObj: file,
      name: file.name,
      size: (file.size / 1024 / 1024).toFixed(2) + " MB",
      pageCount: pdf.numPages,
      pdfDoc: pdf,
    });

    for (let p = 1; p <= pdf.numPages; p++) {
      state.pages.push({
        id: `p-${fileId}-${p}`,
        fileId: fileId,
        pageNum: p,
        rotation: 0,
        pdfDoc: pdf,
      });
    }
  }

  els.stepUpload.classList.add("hidden");
  els.stepEditor.classList.remove("hidden");
  els.stepEditor.classList.add("flex");
  renderGrid();
  hideLoading();
}

// --- VIEW LOGIC ---
function switchView(mode) {
  state.viewMode = mode;
  state.selectedIds.clear();

  if (mode === "files") {
    els.btnViewFiles.classList.add("active");
    els.btnViewPages.classList.remove("active");
    els.zoomControls.classList.add("hidden");
    els.editTools.classList.add("opacity-50", "pointer-events-none"); // Disable rotate in file mode
  } else {
    els.btnViewFiles.classList.remove("active");
    els.btnViewPages.classList.add("active");
    els.zoomControls.classList.remove("hidden");
    els.editTools.classList.remove("opacity-50", "pointer-events-none");
  }
  renderGrid();
}

function renderGrid() {
  els.gridContent.innerHTML = "";

  if (state.viewMode === "files") renderFileView();
  else renderPageView();
}

// --- RENDER FILE VIEW ---
function renderFileView() {
  els.gridContent.className = "file-grid";
  els.itemCount.innerText = `${state.files.length} PDF Files`;
  
  state.files.forEach((file, index) => {
      const isSelected = state.selectedFileId === file.id;
      const div = document.createElement("div");
      div.className = `file-card ${isSelected ? "selected" : ""}`;

    div.draggable = true;
    div.dataset.fileId = file.id;

    div.onclick = (e) => {
      if (e.target.closest("button")) return;
      selectFile(file.id);
    };

    // ✅ Disable drag ONLY when clicking buttons
    div.addEventListener("dragstart", (e) => {
      if (e.target.closest("button")) {
        e.preventDefault();
        return;
      }
      onFileDragStart(e);
    });
    // div.onclick = () => selectFile(file.id);
    // div.addEventListener("dragstart", onFileDragStart);
    div.addEventListener("dragend", onFileDragEnd);
    div.addEventListener("dragover", onFileDragOver);
    div.addEventListener("drop", onFileDrop);

    div.innerHTML = `
  <div class="file-thumb">
    <canvas class="file-thumb-canvas"></canvas>
  </div>

  <div class="file-info">
    <h4 class="file-name">${file.name}</h4>
    <p class="file-meta">${file.pageCount} Pages • ${file.size}</p>
  </div>

  <div class="file-actions">
    <button
     onclick="event.stopPropagation(); openFilePagesModal('${file.id}')"
      class="btn-view-pages">
      View Pages
    </button>

    <button
      onclick="event.stopPropagation(); deleteFile('${file.id}')"
      class="btn-delete-file">
      ✕
    </button>
  </div>
`;


    els.gridContent.appendChild(div);
    renderThumbnail(file.pdfDoc, 1, div.querySelector("canvas"));
  });
}

// --- RENDER PAGE VIEW ---
function renderPageView() {
  els.gridContent.className = `grid-content zoom-${state.zoomLevel}`;

  els.itemCount.innerText = `${state.pages.length} Pages`;

  state.pages.forEach((page, index) => {
    const div = document.createElement("div");
    const isSelected = state.selectedIds.has(page.id);

    div.className = `page-card ${isSelected ? "selected" : ""}`;

    div.draggable = true;
    div.dataset.pageId = page.id;
    div.onclick = () => togglePageSelect(page.id);
    div.addEventListener("dragstart", onPageDragStart);
    div.addEventListener("dragend", onPageDragEnd);
    div.addEventListener("dragover", onPageDragOver);
    div.addEventListener("drop", onPageDrop);

    div.innerHTML = `
  <div class="page-preview">
    <canvas class="page-canvas"
      style="transform: rotate(${page.rotation}deg)"></canvas>

    <div class="page-index">${index + 1}</div>

    <div class="page-hover-actions">
      <button onclick="event.stopPropagation(); rotateOnePage('${page.id}')"
              class="page-action rotate">↻</button>
      <button onclick="event.stopPropagation(); deleteOnePage('${page.id}')"
              class="page-action delete">✕</button>
    </div>
  </div>
`;

    els.gridContent.appendChild(div);
    renderThumbnail(page.pdfDoc, page.pageNum, div.querySelector("canvas"));
  });
}

async function renderThumbnail(pdfDoc, pageNum, canvas) {
  try {
    const page = await pdfDoc.getPage(pageNum);
    const viewport = page.getViewport({ scale: 0.5 });
    canvas.height = viewport.height;
    canvas.width = viewport.width;
    await page.render({
      canvasContext: canvas.getContext("2d"),
      viewport: viewport,
    }).promise;
  } catch (e) {}
}

// --- ACTIONS (Delete/Rotate) ---
function deleteFile(id) {
  state.files = state.files.filter((f) => f.id !== id);
  rebuildPagesFromFileOrder();
  renderGrid();
}

function moveFile(index, direction) {
  if (index + direction < 0 || index + direction >= state.files.length) return;
  const temp = state.files[index];
  state.files[index] = state.files[index + direction];
  state.files[index + direction] = temp;
  rebuildPagesFromFileOrder();
  renderGrid();
}

function rebuildPagesFromFileOrder() {
  let newPages = [];
  state.files.forEach((file) => {
    // Find existing pages belonging to this file
    const filePages = state.pages.filter((p) => p.fileId === file.id);
    if (filePages.length > 0) {
      newPages = [...newPages, ...filePages];
    } else {
      // If mistakenly deleted all, restore them (optional safety)
      for (let p = 1; p <= file.pageCount; p++) {
        newPages.push({
          id: `p-${file.id}-${p}`,
          fileId: file.id,
          pageNum: p,
          rotation: 0,
          pdfDoc: file.pdfDoc,
        });
      }
    }
  });
  state.pages = newPages;
}

// Page Actions
function togglePageSelect(id) {
    if (isDraggingPage || isDraggingFile) return;
  
    state.selectedIds.clear();
    state.selectedIds.add(id);
    state.selectedFileId = null; // clear file selection
  
    renderGrid();
  }
  
  

function rotateOnePage(id) {
  const p = state.pages.find((x) => x.id === id);
  if (p) {
    p.rotation = (p.rotation + 90) % 360;
    renderGrid();
  }
}

function deleteOnePage(id) {
  state.pages = state.pages.filter((x) => x.id !== id);
  state.selectedIds.delete(id);
  renderGrid();
}

// ===================== MANIFEST =====================
function buildPageManifest() {
  return state.pages.map((p) => ({
    fileId: p.fileId,
    pageNum: p.pageNum,
    rotation: p.rotation,
  }));
}

// Bulk Actions
window.rotateSelected = function () {
  state.pages.forEach((p) => {
    if (state.selectedIds.has(p.id)) p.rotation = (p.rotation + 90) % 360;
  });
  renderGrid();
};

window.deleteSelected = function () {
  state.pages = state.pages.filter((p) => !state.selectedIds.has(p.id));
  state.selectedIds.clear();
  renderGrid();
};

window.zoom = function (dir) {
  if (dir === "in" && state.zoomLevel < 2) state.zoomLevel++;
  if (dir === "out" && state.zoomLevel > 0) state.zoomLevel--;
  renderGrid();
};

function showLoading() {
  els.loadingOverlay.classList.remove("hidden");
  els.loadingOverlay.classList.add("flex");
}
function hideLoading() {
  els.loadingOverlay.classList.add("hidden");
  els.loadingOverlay.classList.remove("flex");
}


// ===================== APPLY =====================
els.applyBtn.onclick = async () => {
  if (!state.pages.length) {
    alert("No pages to merge");
    return;
  }

  try {
    showLoading();

    const formData = new FormData();

    state.files.forEach((f) => {
      formData.append("files", f.fileObj);
      formData.append("fileIds", f.id);
    });

    formData.append(
      "pageManifest",
      JSON.stringify(buildPageManifest())
    );

    formData.append("quality", state.quality);

    const res = await fetch("/merge-pdfs", {
      method: "POST",
      body: formData,
    });

    if (!res.ok) {
      throw new Error("Merge failed");
    }

    const blob = await res.blob();
    const url = URL.createObjectURL(blob);

    els.downloadBtn.href = url;
    els.downloadBtn.download =
      (els.outputFilename.value || "merged") + ".pdf";

    els.stepEditor.classList.add("hidden");
    els.stepSuccess.classList.remove("hidden");
  } catch (err) {
    console.error(err);
    alert(err.message);
  } finally {
    hideLoading();
  }
};


function reorderFiles(sourceId, targetId) {
  const srcIndex = state.files.findIndex((f) => f.id === sourceId);
  const tgtIndex = state.files.findIndex((f) => f.id === targetId);

  const [movedFile] = state.files.splice(srcIndex, 1);
  state.files.splice(tgtIndex, 0, movedFile);

  rebuildPagesFromFileOrder();
  renderGrid();
}

function moveSelected(direction) {
    if (state.viewMode === "pages") {
      reorderSelectedPage(direction);
    } else {
      reorderSelectedFile(direction);
    }
}

function reorderPages(sourceId, targetId) {
  const srcIndex = state.pages.findIndex((p) => p.id === sourceId);
  const tgtIndex = state.pages.findIndex((p) => p.id === targetId);

  const [moved] = state.pages.splice(srcIndex, 1);
  state.pages.splice(tgtIndex, 0, moved);

  renderGrid();
}

function onPageDragStart(e) {
  draggedPageId = e.currentTarget.dataset.pageId;
  state.selectedIds.clear();
  state.selectedIds.add(draggedPageId);


  const preview = createDragPreview(e.currentTarget);
  e.dataTransfer.setDragImage(
    preview,
    preview.offsetWidth / 2,
    preview.offsetHeight / 2
  );

  e.currentTarget.classList.add("opacity-50");

  // cleanup preview
  setTimeout(() => preview.remove(), 0);
}

function onPageDragOver(e) {
  e.preventDefault();
  e.currentTarget.classList.add("drag-over");
}

function onPageDrop(e) {
  e.preventDefault();
  e.currentTarget.classList.remove("drag-over");

  const targetPageId = e.currentTarget.dataset.pageId;
  if (!draggedPageId || draggedPageId === targetPageId) return;

  reorderPages(draggedPageId, targetPageId);
}

function onPageDragEnd(e) {
  e.currentTarget.classList.remove("opacity-50");
  document
    .querySelectorAll(".page-card.drag-over")
    .forEach((el) => el.classList.remove("drag-over"));
  draggedPageId = null;
}

function createDragPreview(sourceEl) {
  const clone = sourceEl.cloneNode(true);
  clone.style.position = "absolute";
  clone.style.top = "-1000px";
  clone.style.left = "-1000px";
  clone.style.width = sourceEl.offsetWidth + "px";
  clone.style.height = sourceEl.offsetHeight + "px";
  clone.style.opacity = "0.9";
  clone.style.pointerEvents = "none";
  document.body.appendChild(clone);
  return clone;
}

function onFileDragStart(e) {
    draggedFileId = e.currentTarget.dataset.fileId;
    isDraggingFile = true;
  
    const preview = createDragPreview(e.currentTarget);
    e.dataTransfer.setDragImage(
      preview,
      preview.offsetWidth / 2,
      preview.offsetHeight / 2
    );
  
    e.currentTarget.classList.add("opacity-50");
    setTimeout(() => preview.remove(), 0);
  }
  
  function onFileDragOver(e) {
    e.preventDefault();
    e.currentTarget.classList.add("drag-over");
  }
  
  function onFileDrop(e) {
    e.preventDefault();
    e.currentTarget.classList.remove("drag-over");
  
    const targetId = e.currentTarget.dataset.fileId;
    if (!draggedFileId || draggedFileId === targetId) return;
  
    reorderFiles(draggedFileId, targetId);
  }
  
  function onFileDragEnd(e) {
    e.currentTarget.classList.remove("opacity-50");
    document
      .querySelectorAll(".file-card.drag-over")
      .forEach((el) => el.classList.remove("drag-over"));
  
    draggedFileId = null;
    isDraggingFile = false;
  }
  
  function reorderSelectedPage(direction) {
    if (state.selectedIds.size !== 1) {
      alert("Select one page first");
      return;
    }
  
    const pageId = [...state.selectedIds][0];
    const index = state.pages.findIndex(p => p.id === pageId);
    if (index === -1) return;
  
    const page = state.pages[index];
    state.pages.splice(index, 1);
  
    switch (direction) {
      case "up":
        state.pages.splice(Math.max(0, index - 1), 0, page);
        break;
      case "down":
        state.pages.splice(Math.min(state.pages.length, index + 1), 0, page);
        break;
      case "top":
        state.pages.unshift(page);
        break;
      case "end":
        state.pages.push(page);
        break;
    }
  
    state.selectedIds.clear();
    state.selectedIds.add(page.id);
  
    renderGrid();
  }
  

  function reorderSelectedFile(direction) {
    if (!state.selectedFileId) {
      alert("Select one file first");
      return;
    }
  
    const index = state.files.findIndex(f => f.id === state.selectedFileId);
    if (index === -1) return;
  
    const file = state.files[index];
    state.files.splice(index, 1);
  
    switch (direction) {
      case "up":
        state.files.splice(Math.max(0, index - 1), 0, file);
        break;
      case "down":
        state.files.splice(Math.min(state.files.length, index + 1), 0, file);
        break;
      case "top":
        state.files.unshift(file);
        break;
      case "end":
        state.files.push(file);
        break;
    }
  
    rebuildPagesFromFileOrder();
    state.selectedFileId = file.id;
  
    renderGrid();
  }
  
  
  function selectFile(fileId) {
    state.selectedFileId = fileId;
    state.selectedIds.clear(); // clear page selection
    renderGrid();
  }
 // ===================== SPLIT MODAL LOGIC =====================

// ===================== MODAL LOGIC WITH SAVE BUTTON =====================

let currentModalFileId = null;
let activeModalPageId = null;
let tempModalPages = []; // Temporary storage for edits

const modal = document.getElementById("filePagesModal");
const modalPageList = document.getElementById("modalPageList");
const modalPreviewContainer = document.getElementById("modalPreviewContainer");
const modalTitle = document.getElementById("modalTitle");

// --- OPEN MODAL (Initialize Temp State) ---
// --- OPEN MODAL (Initialize Temp State) ---
function openFilePagesModal(fileId) {
  currentModalFileId = fileId;
  const file = state.files.find(f => f.id === fileId);
  if (!file) return;

  modalTitle.innerText = `Editing – ${file.name}`;
  
  // === FIX START ===
  // ERROR CAUSE: JSON.stringify crashes on pdfDoc (circular reference).
  // SOLUTION: Use .map() and object spread (...) to create a clone manually.
  
  tempModalPages = state.pages
      .filter(p => p.fileId === fileId)
      .map(p => ({
          ...p,             // Copies id, rotation, pageNum (Values)
          pdfDoc: p.pdfDoc  // Copies reference to PDF (Safe, we don't edit this)
      }));
  // === FIX END ===

  // Select first page
  if (tempModalPages.length > 0) {
      activeModalPageId = tempModalPages[0].id;
  } else {
      activeModalPageId = null;
  }

  renderModalLayout();
  
  document.body.classList.add("modal-open");
  modal.classList.remove("hidden");
  modal.classList.add("flex");
}

// --- CLOSE MODAL (Discard Changes) ---
function closeFilePagesModal() {
    document.body.classList.remove("modal-open");
    modal.classList.add("hidden");
    modal.classList.remove("flex");
    
    // Reset temp variables
    currentModalFileId = null;
    activeModalPageId = null;
    tempModalPages = [];
}

// --- SAVE CHANGES (Commit to State) ---
window.saveModalChanges = function() {
  if (!currentModalFileId) return;

  // 1. Delete Logic: Jo pages Temp me nahi hain, unhe Main State se uda do
  const keptPageIds = new Set(tempModalPages.map(p => p.id));
  
  state.pages = state.pages.filter(p => {
      // Agar page isi file ka hai, to check karo ki wo kept list me hai ya nahi
      if (p.fileId === currentModalFileId) {
          return keptPageIds.has(p.id);
      }
      return true; // Baaki files ke pages ko touch mat karo
  });

  // 2. Rotate Logic: Kept pages ki rotation update karo
  tempModalPages.forEach(tempP => {
      const realPage = state.pages.find(p => p.id === tempP.id);
      if (realPage) {
          realPage.rotation = tempP.rotation;
      }
  });

  // === FIX START: PAGE COUNT UPDATE ===
  // 3. File ka Page Count update karo (Taaki Grid me sahi number dikhe)
  const fileIndex = state.files.findIndex(f => f.id === currentModalFileId);
  
  if (fileIndex !== -1) {
      // Count karo ki ab is file ke kitne pages bache hain
      const remainingPagesCount = state.pages.filter(p => p.fileId === currentModalFileId).length;

      if (remainingPagesCount === 0) {
          // Agar ek bhi page nahi bacha, to poori file uda do
          state.files = state.files.filter(f => f.id !== currentModalFileId);
      } else {
          // Update the count property
          state.files[fileIndex].pageCount = remainingPagesCount;
      }
  }
  // === FIX END ===

  // 4. UI Refresh
  renderGrid(); 
  closeFilePagesModal();
};

// --- RENDER FUNCTIONS (Use tempModalPages) ---

function renderModalLayout() {
    renderSidebar();
    renderPreview();
}

function renderSidebar() {
    modalPageList.innerHTML = "";
    
    // Use TEMP pages
    tempModalPages.forEach((page, index) => {
        const isActive = page.id === activeModalPageId;
        
        const div = document.createElement("div");
        div.className = `modal-page-item ${isActive ? "active" : ""}`;
        
        div.onclick = () => {
            if(activeModalPageId === page.id) return;
            activeModalPageId = page.id;
            renderModalLayout();
        };

        div.innerHTML = `
            <button onclick="deletePageFromSidebar(event, '${page.id}')"
                    class="modal-delete-btn">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
            </button>

            <div class="modal-thumb">
                 <canvas class="modal-thumb-canvas"
                         style="transform: rotate(${page.rotation}deg)"></canvas>
            </div>
            
            <div class="modal-page-index">${index + 1}</div>
        `;

        modalPageList.appendChild(div);
        renderThumbnail(page.pdfDoc, page.pageNum, div.querySelector("canvas"));
        
        if(isActive) {
            setTimeout(() => div.scrollIntoView({ behavior: 'smooth', block: 'nearest' }), 50);
        }
    });
}

async function renderPreview() {
    const previewContainer = document.getElementById("modalPreviewContainer");
    const controls = document.getElementById("modalOverlayControls");
    
    // Cleanup old canvas
    const existingCanvas = previewContainer.querySelector('canvas');
    if (existingCanvas) existingCanvas.remove();

    // Use TEMP pages
    const page = tempModalPages.find(p => p.id === activeModalPageId);
    
    if (!page) {
        if(controls) controls.classList.add('hidden');
        return;
    }
    if(controls) controls.classList.remove('hidden');

    const canvas = document.createElement('canvas');
    canvas.className = "modal-preview-canvas";
    canvas.style.transform = `rotate(${page.rotation}deg)`;
    previewContainer.insertBefore(canvas, controls);

    try {
        const pdfPage = await page.pdfDoc.getPage(page.pageNum);
        const viewport = pdfPage.getViewport({ scale: 1.5 });
        canvas.height = viewport.height;
        canvas.width = viewport.width;

        await pdfPage.render({
            canvasContext: canvas.getContext("2d"),
            viewport: viewport,
        }).promise;
    } catch (e) { console.error(e); }
}


// --- ACTIONS (Modify TEMP State) ---

window.deletePageFromSidebar = function(e, pageId) {
    e.stopPropagation();
    
    // Modify TEMP array
    const deleteIndex = tempModalPages.findIndex(p => p.id === pageId);
    if (deleteIndex === -1) return;

    tempModalPages.splice(deleteIndex, 1);

    // Smart Selection Logic (Next/Prev)
    if (activeModalPageId === pageId) {
        if (tempModalPages.length > 0) {
            let newIndex = deleteIndex;
            if (newIndex >= tempModalPages.length) newIndex = tempModalPages.length - 1;
            activeModalPageId = tempModalPages[newIndex].id;
        } else {
            activeModalPageId = null;
        }
    }

    renderModalLayout();
    // NOTE: We do NOT call renderGrid() here. 
    // Changes are only applied when user clicks SAVE.
};

window.rotateActivePage = function() {
    if (!activeModalPageId) return;
    const page = tempModalPages.find(p => p.id === activeModalPageId);
    if (page) {
        page.rotation = (page.rotation + 90) % 360;
        renderModalLayout();
    }
};

window.deleteActivePage = function() {
    if (!activeModalPageId) return;
    window.deletePageFromSidebar({ stopPropagation: () => {} }, activeModalPageId);
};