let imageFiles = [];
let selectedIdx = -1;
let globalSettings = { orientation: 'p', size: 'fit', margin: 0 };
let sortableInstance = null; // Track sortable instance

document.addEventListener("DOMContentLoaded", function () {
    //document.getElementById('fileInput').addEventListener("change", (e) => handleFileSelect(e.target.files));
    document.getElementById('fileInput').addEventListener("change", handleFileSelect);
});


// Initialize Sortable on the grid
function initSortable() {
    const grid = document.getElementById('gridContent');
    if (sortableInstance) sortableInstance.destroy();

    sortableInstance = new Sortable(grid, {
        animation: 150,
        ghostClass: 'sortable-ghost',
        onEnd: function () {
            // Reorder the imageFiles array based on the new DOM order
            const newOrder = Array.from(grid.children).map(card => parseInt(card.dataset.index));
            const reorderedFiles = newOrder.map(index => imageFiles[index]);
            
            imageFiles = reorderedFiles;
            selectedIdx = -1; // Reset selection to avoid confusion
            renderImages();   // Re-render to update the "Page X" labels
        }
    });
}

function handleFileSelect(e) {
    const files = Array.from(e.target.files);
    let loadedCount = 0;

    files.forEach(file => {
        if (file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = (event) => {
                imageFiles.push({
                    src: event.target.result,
                    name: file.name,
                    rotation: 0, // Initialize rotation at 0 degrees
                    settings: { ...globalSettings }
                });
                loadedCount++;
                if (loadedCount === files.length) {
                    renderImages();
                }
            };
            reader.readAsDataURL(file);
        }
    });
    document.getElementById('step-upload').classList.add('hidden');
    document.getElementById('step-editor').classList.remove('hidden');
}

function selectCard(index) {
    selectedIdx = (selectedIdx === index) ? -1 : index;
    updateUI();
    renderImages();
}

function deselectAll() {
    selectedIdx = -1;
    updateUI();
    renderImages();
}

function applySetting(type, value) {
    if (selectedIdx !== -1) {
        // Apply to specific image
        imageFiles[selectedIdx].settings[type] = value;
    } else {
        // Apply to everything (Global Mode)
        globalSettings[type] = value;
        imageFiles.forEach(img => {
            img.settings[type] = value;
        });
    }
    updateUI();     // Updates button highlights
    renderImages(); // Refreshes the grid visually
}

function updateUI() {
    // Determine which settings to show in the sidebar
    const s = (selectedIdx !== -1) ? imageFiles[selectedIdx].settings : globalSettings;
    
    document.getElementById('statusText').innerText = (selectedIdx !== -1) ? `Editing Image ${selectedIdx + 1}` : "Global Mode (All)";
    
    // Update Orientation Buttons
    document.getElementById('btn-p').classList.toggle('active', s.orientation === 'p');
    document.getElementById('btn-l').classList.toggle('active', s.orientation === 'l');
    
    // Update Dropdown
    document.getElementById('pageSize').value = s.size;
    
    // Update Margin Buttons
    [0, 15, 30, 50].forEach(m => {
        const btn = document.getElementById(`m-${m}`);
        if(btn) btn.classList.toggle('active', s.margin === m);
    });
}

function renderImages() {
    const grid = document.getElementById('gridContent');
    grid.innerHTML = '';
    
    imageFiles.forEach((img, index) => {
        const s = img.settings;
        const card = document.createElement('div');
        
        // Match selection style
        const isSelected = selectedIdx === index ? 'selected' : '';
        card.className = `image-card ${isSelected}`;
        card.dataset.index = index;
        card.onclick = () => selectCard(index);

        // DYNAMIC CLASS GENERATION: canvas + size + orientation (e.g., canvas-a4-p)
        const canvasSizeClass = `canvas-${s.size}-${s.orientation}`;

        card.innerHTML = `
            <div class="card-actions">
                <button title="Rotate" onclick="event.stopPropagation(); rotateSingle(${index})" class="action-btn action-primary">
                    <svg class="icon-sm" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/></svg>
                </button>
                <button title="Remove" onclick="event.stopPropagation(); removeImage(${index})" class="action-btn action-muted">
                    <svg class="icon-sm" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                </button>
            </div>

           
            <div class="paper-canvas ${canvasSizeClass}" style="padding: ${s.margin / 4}px">
                <img src="${img.src}" 
                    alt="${img.name}" 
                    class="preview-img"
                    style="transform: rotate(${img.rotation}deg);" 
                    onclick="event.stopPropagation(); openModal(${index})">
            </div>
    
            
            <div class="filename-label" title="${img.name}">
                ${img.name}
            </div>
        `;
        grid.appendChild(card);
    });
    
    document.getElementById('itemCount').innerText = `${imageFiles.length} Images`;
    initSortable(); 
}

function rotateSingle(index) {
    // Increment rotation by 90 and reset to 0 if it hits 360
    imageFiles[index].rotation = (imageFiles[index].rotation + 90) % 360;
    
    // Also toggle orientation setting automatically for better UX
    // If it's 90 or 270, it's effectively Landscape (l)
    const r = imageFiles[index].rotation;
    imageFiles[index].settings.orientation = (r === 90 || r === 270) ? 'l' : 'p';
    
    renderImages();
}

function removeImage(index) {
    imageFiles.splice(index, 1);
    selectedIdx = -1;
    renderImages();
}

function move(index, dir) {
    const next = index + dir;
    if (next >= 0 && next < imageFiles.length) {
        [imageFiles[index], imageFiles[next]] = [imageFiles[next], imageFiles[index]];
        selectedIdx = -1;
        renderImages();
    }
}

async function generatePDF() {
  const { jsPDF } = window.jspdf;
  let pdf = null;

  for (let i = 0; i < imageFiles.length; i++) {
      const imgData = imageFiles[i];
      const s = imgData.settings;

      const format = s.size === 'fit' ? 'a4' : s.size;
      const orient = s.orientation === 'p' ? 'portrait' : 'landscape';

      if (i === 0) {
          pdf = new jsPDF({
              orientation: orient,
              unit: 'mm',
              format: format
          });
      } else {
          pdf.addPage(format, orient);
      }

      const pageWidth = pdf.internal.pageSize.getWidth();
      const pageHeight = pdf.internal.pageSize.getHeight();
      const margin = s.margin;

      const maxW = pageWidth - margin * 2;
      const maxH = pageHeight - margin * 2;

      // Load image to get natural dimensions
      const img = new Image();
      img.src = imgData.src;

      await new Promise(resolve => {
          img.onload = () => {
              const imgW = img.width;
              const imgH = img.height;

              const imgRatio = imgW / imgH;
              const pageRatio = maxW / maxH;

              let drawW, drawH;

              if (imgRatio > pageRatio) {
                  // Image is wider
                  drawW = maxW;
                  drawH = maxW / imgRatio;
              } else {
                  // Image is taller
                  drawH = maxH;
                  drawW = maxH * imgRatio;
              }

              // Center image
              const x = (pageWidth - drawW) / 2;
              const y = (pageHeight - drawH) / 2;

              const imgType = imgData.src.startsWith('data:image/png')
                  ? 'PNG'
                  : 'JPEG';

              pdf.addImage(
                  imgData.src,
                  imgType,
                  x,
                  y,
                  drawW,
                  drawH
              );

              resolve();
          };
      });
  }
  document.getElementById('downloadBtn').onclick = e => {
    e.preventDefault();
    pdf.save(
        (document.getElementById('outputFilename').value || 'PDF_NEX') + '.pdf'
    );
};

document.getElementById('step-editor').classList.add('hidden');
document.getElementById('step-success').classList.remove('hidden');
}

function openModal(index) {
    const modal = document.getElementById('previewModal');
    const modalImg = document.getElementById('modalImage');
    modalImg.src = imageFiles[index].src;
    modal.classList.remove('hidden');
    // Prevent scrolling behind modal
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    const modal = document.getElementById('previewModal');
    modal.classList.add('hidden');
    document.body.style.overflow = 'auto';
}

// Optional: Close modal when clicking outside the image
document.getElementById('previewModal').onclick = function(e) {
    if (e.target === this) closeModal();
};